// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef THIRD_PARTY_WEBRTC_OVERRIDES_P2P_BASE_ICE_INTERACTION_INTERFACE_H_
#define THIRD_PARTY_WEBRTC_OVERRIDES_P2P_BASE_ICE_INTERACTION_INTERFACE_H_

#include "base/containers/span.h"
#include "base/memory/ref_counted.h"

#include "third_party/webrtc/api/rtc_error.h"
#include "third_party/webrtc/rtc_base/system/rtc_export.h"
#include "third_party/webrtc_overrides/p2p/base/ice_connection.h"
#include "third_party/webrtc_overrides/p2p/base/ice_ping_proposal.h"
#include "third_party/webrtc_overrides/p2p/base/ice_prune_proposal.h"
#include "third_party/webrtc_overrides/p2p/base/ice_switch_proposal.h"

namespace blink {

// An interface that supports interactions with an ICE controller in response to
// events generated by the ICE controller or initiated by an external observer.
class RTC_EXPORT IceInteractionInterface
    : public base::RefCounted<IceInteractionInterface> {
 public:
  virtual ~IceInteractionInterface() = default;

  // Accept a proposal to send a STUN ping on an ICE connection.
  virtual void AcceptPingProposal(const IcePingProposal& ping_proposal) = 0;
  // Reject a proposal to send a STUN ping on an ICE connection.
  virtual void RejectPingProposal(const IcePingProposal& ping_proposal) = 0;

  // Accept a proposal to switch the ICE transport to an ICE connection.
  virtual void AcceptSwitchProposal(
      const IceSwitchProposal& switch_proposal) = 0;
  // Reject a proposal to switch the ICE transport to an ICE connection.
  virtual void RejectSwitchProposal(
      const IceSwitchProposal& switch_proposal) = 0;

  // Accept a proposal to prune away certain ICE connections.
  virtual void AcceptPruneProposal(const IcePruneProposal& prune_proposal) = 0;
  // Reject a proposal to prune away certain ICE connections.
  virtual void RejectPruneProposal(const IcePruneProposal& prune_proposal) = 0;

  // Send a STUN ping on the ICE connection.
  virtual webrtc::RTCError PingIceConnection(
      const IceConnection& connection) = 0;
  // Switch the ICE transport to the ICE connection.
  virtual webrtc::RTCError SwitchToIceConnection(
      const IceConnection& connection) = 0;
  // Prune the supplied ICE connections.
  virtual webrtc::RTCError PruneIceConnections(
      base::span<const IceConnection> connections_to_prune) = 0;
};

}  // namespace blink

#endif  // THIRD_PARTY_WEBRTC_OVERRIDES_P2P_BASE_ICE_INTERACTION_INTERFACE_H_
